%[text] # User Scenarios
%%
%[text] #### User Scenario 1
  %[control:button:8f79]{"position":[1,2]}
% at a constant rate of 0.3 degrees per step:
% from 0.4 to 1.6 minutes, move the head down 
% from 1.6 to 2.8 minutes move the head up 
% and from 6.4 to 7.6 minutes, move the head down 
% and from 7.6 to 8.8 minutes, move the head up 

CreateScenario(40,160,"Down",0.3,160,280,"Up",0.3,640,760,"Down",0.3,760,880,"Up",0.3);

%%
%[text] #### User Scenario 2
  %[control:button:99b5]{"position":[1,2]}

% at a constant rate of 0.3 degrees per step
% From 0.4 to 1.4 minutes, move the head down, then stay there
CreateScenario(40,140,"Down",0.3);
%%
%[text] #### User Scenario 3
  %[control:button:1bcd]{"position":[1,2]}

% Move down from 0.4 to 1.4 min, then up from 3.8 to 4.8 min
% Move down faster from 4.8 to 5.8 min, then up slower from 7.2 to 8.2 min

CreateScenario(...
    40,140,"Down",0.3,...
    380,480,"Up",0.3,...
    480,580,"Down",0.5,...
    720,820,"Up",0.3);
%%
%[text] #### User Scenario 4
  %[control:button:488c]{"position":[1,2]}

CreateScenario(...
    0,100,"Down",0.35,...
    100,199,"Up",0.3,...
    200,299,"Down",0.3,...
    300,399,"Up",0.3,...
    400,499,"Down",0.3,...
    500,599,"Up",0.3,...
    600,699,"Down",0.3,...
    700,799,"Up",0.3,...
    800,899,"Down",0.3,...
    900,999,"Up",0.3,...
    1000,1000,"Down",0.3);
%%
function TextNeckPlots = SetUpPlots(TotTimeSteps)
% SETUPPLOTS Function to set up 3D plots for torso and head, and a 2D plot for text neck over time
%
% Output Arguments:
%     TextNeckPlots - structure containing handles for the plots
%         .CurrPlot - handle for the current plot of text neck over time
%         .SensorPlot - handle for the sensor plot
%         .TextNeckAxes - axes for the text neck plot
%         .head - handle for the head plot
%         .torso - handle for the torso plot

clf % Clear current figure
f = figure('Position', [0 0 1100 400]); % Create a new figure with specified position
t = tiledlayout(1,2); % Create a tiled layout for multiple plots
nexttile; % Move to the next tile in the layout
% Plot torso as a 3D line
torso = plot3([4 4], [4 4], [4 8], 'SeriesIndex', 1, 'LineWidth', 3);
hold on % Hold the current plot to overlay additional plots
% Plot arms as a 3D line
arms = plot3([3 2.5 3 4 5 5.5 5], zeros(1, 7) + 4, [5 6 7 7 7 6 5], 'SeriesIndex', 1, 'LineWidth', 3);
% Plot legs as a 3D line
legs = plot3([3 3 3 4 5 5 5], [3.5, zeros(1, 5) + 4, 3.5], [1 1 3 4 3 1 1], 'SeriesIndex', 1, 'LineWidth', 3);
theta = linspace(0, 2*pi, 1000); % Create an array of angles for the head
x = 0.5 * cos(theta) + 4; % X coordinates for the head
y = zeros(1, length(theta)) + 4; % Y coordinates for the head
z = sin(theta) + 9; % Z coordinates for the head
% Plot head as a 3D line
head = plot3(x, y, z, 'SeriesIndex', 1, 'LineWidth', 3);
MyColors = colororder; % Get the current color order
% Fill areas with transparency to represent different sections
fill3([0 0 10 10], [0 10 10 0], [1 1 1 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([10 10 10 10], [0 10 10 0], [1 1 10 10], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([0 0 10 10], [10 10 10 10], [1 10 10 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
xlim([0 10]) % Set limits for the x-axis
ylim([0 10]) % Set limits for the y-axis
zlim([0 11]) % Set limits for the z-axis
hold off
ax = gca; % Get current axes
ax.Visible = 'off'; % Hide the axes
SensorPlot = PlotSensor(torso); % Call function to plot sensor data

% Set up the axes for the text neck plot
TextNeckAxes = axes(t);
TextNeckAxes.Layout.Tile = 2;   % Position the axes in the second tile
CurrPlot = plot(0, false, 'b'); % Initialize the current plot
xlim([0 TotTimeSteps]);         % Set limits of the text neck plot in 1/100 minute increments
ylim([-0.25 1.25]);             % Set limits of the text neck plot
xlabel('Minutes');              % Label for the x-axis
ylabel('Text Neck?');           % Label for the y-axis
title('Text Neck Over Time');   % Title for the plot
yticks([0 1]);                  % Set y-ticks
xticks(0:100:TotTimeSteps);     % Set x-ticks
xticklabels(0:1:10);            % Set x-tick labels
xtickangle(0);                  % Set angle for x-tick labels
yticklabels({'No', 'Yes'});     % Set y-tick labels

% Create output structure
TextNeckPlots.CurrPlot = CurrPlot;
TextNeckPlots.SensorPlot = SensorPlot;
TextNeckPlots.TextNeckAxes = TextNeckAxes;
TextNeckPlots.head = head;
TextNeckPlots.torso = torso;
end

function TNData = InitializeTextNeckData(TotTimeSteps)
TNData.ContiguousTimeInTNP = 0;
TNData.TextNeck = false;
TNData.History = zeros([1 TotTimeSteps]);
end

function SensorPlot = PlotSensor(torso)
hold on
SensorPlot = scatter3(torso.XData(2), torso.YData(2), torso.ZData(2), 'SizeData', 30, 'SeriesIndex', 'none');
hold off

ax = gca;
ax.Visible = 'off';
end

function CreateScenario(Start,Stop,Mode,Speed)
arguments (Repeating)
    Start (1,1) double {mustBeInteger}
    Stop (1,1) double {mustBeInteger}
    Mode (1,1) string {mustBeMember(Mode,["Up","Down"])}
    Speed (1,1) double
end
% Note that Start and Stop are timestep indices, not times
TotTimeSteps = 1000; % With step size 0.01 min
TNPlot = SetUpPlots(TotTimeSteps);
TNData = InitializeTextNeckData(TotTimeSteps);

TopOfSpine = [TNPlot.torso.XData(2), TNPlot.torso.YData(2), TNPlot.torso.ZData(2)];
MoveIdx = 1;

if any([Stop{:}]-[Start{:}]<0)
    error("Wrong direction!")
    return
end

for timeStep = 1:TotTimeSteps % in 0.01 minute increments
    if MoveIdx <= length(Start)
        if timeStep == Stop{MoveIdx}
            MoveIdx = MoveIdx + 1; % Move to the next scenario
        elseif (timeStep >= Start{MoveIdx}) && (timeStep < Stop{MoveIdx})
            % Move the head down or up based on the mode
            if Mode{MoveIdx} == "Down"
                rotate(TNPlot.head, [1 0 0], Speed{MoveIdx}, TopOfSpine);
            else % Move up
                rotate(TNPlot.head, [1 0 0], -Speed{MoveIdx}, TopOfSpine);
            end
        end
    end
    [TNPlot,TNData] = CheckPosition(timeStep, TNPlot, TNData);
    if TNData.History(timeStep)
        patch([timeStep-1 timeStep-1 timeStep timeStep], [-0.25 1.25 1.25 -0.25], 'red', 'EdgeColor', 'none', 'FaceAlpha', 0.2, 'Parent', TNPlot.TextNeckAxes);
    end
end
TNState = TNData.History;
end

function TextNeck = DetermineNeckPosition(TNPlot,TNAngle)
arguments
    TNPlot 
    TNAngle (1,1) double {mustBeInteger,mustBePositive} = 20 
end
% Calculate vectors from torso to the top of the head
% 250/1000 is one quarter of the way around
% and between torso points to compute the neck angle
Vec1 = [TNPlot.head.XData(250) TNPlot.head.YData(250) TNPlot.head.ZData(250)] - [TNPlot.torso.XData(2) TNPlot.torso.YData(2) TNPlot.torso.ZData(2)];
Vec2 = [TNPlot.torso.XData(1) TNPlot.torso.YData(1) TNPlot.torso.ZData(1)] - [TNPlot.torso.XData(2) TNPlot.torso.YData(2) TNPlot.torso.ZData(2)];

% Compute the angle of the neck in degrees
% Note that standing straight is a neck angle of 180
angle = atan2d(norm(cross(Vec1,Vec2)), dot(Vec1,Vec2));

% Compute TextNeck values for the current position
if (angle > (180+TNAngle)) || (angle < (180-TNAngle))
    % Set TextNeck=true if the neck is bent more than 20 degrees
    TextNeck = true;
    % If the neck angle moves within 20 degrees of 180
elseif (angle <= (180+TNAngle)) && (angle >= (180-TNAngle))
    TextNeck = false;
end
end

function [TNPlot,TNData] = CheckPosition(i, TNPlot, TNData)
% CHECKPOSITION Function to check the position of the neck
%
% Input Arguments:
%     i      - Current timestep index (0.01 minutes each)
%     TNPlot - Structure containing 
%        CurrPlot - a line plot of the textneck status out/0 vs in/1 
%        SensorPlot - a scatter showing where the sensor is located (at the
%              head/neck point on the figure)
%        TextNeckAxes - a handle to the axes containing CurrPlot
%        head - a handle to the line plot of the figure's head
%        torso - a handle to the line plot of the figure's torso
%     TNData - Structure containing data related to neck position
%        ContiguousTimeInTNP - the time in TextNeck without leaving
%        TextNeck - the current state, true is in TextNeck position
%        History - an in/out record for TextNeck position
%
% Output Arguments:
%     TNPlot - Updated plot data structure
%     TNData - Updated data structure with TextNeck History

% Update TNData value at time i

% First, calculate whether we are in TextNeck at all
TextNeck = DetermineNeckPosition(TNPlot);

TNData.History(i) = double(TextNeck);

TNPlot.CurrPlot.XData = [TNPlot.CurrPlot.XData i];
TNPlot.CurrPlot.YData = [TNPlot.CurrPlot.YData TNData.History(i)];

% % Pause for a short duration every 0.2 minutes
% if mod(i,10) == 0
%     pause(0.01)
% end
% Ensure the plot is updated with the latest data
if mod(i,10) == 0
    pause(0.05);
end
end

%[appendix]{"version":"1.0"}
%---
%[metadata:view]
%   data: {"layout":"hidecode","rightPanelPercent":40}
%---
%[control:button:8f79]
%   data: {"label":"Run scenario 1","run":"Section"}
%---
%[control:button:99b5]
%   data: {"label":"Run scenario 2","run":"Section"}
%---
%[control:button:1bcd]
%   data: {"label":"Run scenario 3","run":"Section"}
%---
%[control:button:488c]
%   data: {"label":"Run scenario 4","run":"Section"}
%---
