%% Analog input synchronization using Trigger Bus on Two DT8824 instruments
%   
%  Copyright (C) 2010 DataTranslation Inc.
%

%% Introduction
% This example demonstrates how to synchronize the trigger on two devices using
% the trigger bus and perform continuous anlaog input on both devices.


%% Create a device object for each instrument
% RsrcName is an IVI logical name or an instrument specific string that 
% identifies the address of the instrument, such as a VISA resource descriptor
% string.

masterDevRsrcName = 'TCPIP::192.43.218.69::SOCKET';
slaveDevRsrcName = 'TCPIP::192.43.218.93::SOCKET';

masterDeviceObj = icdevice('DT8824_DT8824.mdd', masterDevRsrcName);
slaveDeviceObj = icdevice('DT8824_DT8824.mdd',slaveDevRsrcName);

try
    %% Connect the device object to the instrument
    
    connect(masterDeviceObj);
    connect(slaveDeviceObj);
    
    %% Get the identity for each instrument
    
    % Get master instrument identity
    comobj = get(masterDeviceObj, 'Identity');
    propertyValue = get(comobj, 'InstrumentModel');
    str = strcat ('InstrumentModel= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'InstrumentManufacturer');
    str = strcat ('InstrumentManufacturer= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'InstrumentFirmwareRevision');
    str = strcat ('InstrumentFirmwareRevision= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Description');
    str = strcat ('Description= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Identifier');
    str = strcat ('Identifier= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Vendor');
    str = strcat ('Vendor= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Revision');
    str = strcat ('Revision= ',propertyValue);  
    disp(str);
    
    % Get slave instrument identity
    comobj = get(slaveDeviceObj, 'Identity');
    propertyValue = get(comobj, 'InstrumentModel');
    str = strcat ('InstrumentModel= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'InstrumentManufacturer');
    str = strcat ('InstrumentManufacturer= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'InstrumentFirmwareRevision');
    str = strcat ('InstrumentFirmwareRevision= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Description');
    str = strcat ('Description= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Identifier');
    str = strcat ('Identifier= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Vendor');
    str = strcat ('Vendor= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Revision');
    str = strcat ('Revision= ',propertyValue);  
    disp(str);

    %% Configure the master and slave instruments
    
    % Abort current acquisition on the master
    groupObj = get(masterDeviceObj, 'Analoginputacquisition');
    masterAnalogInputAcquisitionObj = groupObj(1);
    invoke(masterAnalogInputAcquisitionObj, 'Abort');
    
    % Enable analog input channel 1 for scanning
    %set(masterDeviceObj.Analoginputchannel(1), 'Enabled', 'on');
    
    % Disable wrapping, so that data does not get overwritten in the
    % hardware FIFO when the FIFO is full
    %set(masterDeviceObj.Analoginputacquisition(1), 'WrapEnabled', 'off');
  
    % Abort current acquisition on the slave
    groupObj = get(slaveDeviceObj, 'Analoginputacquisition');
    slaveAnalogInputAcquisitionObj = groupObj(1);
    invoke(slaveAnalogInputAcquisitionObj, 'Abort');
    
    % Enable channel 1 for scanning
   % set(slaveDeviceObj.Analoginputchannel(1), 'Enabled', 'on');

    % Disable wrapping, so that data does not get overwritten in the
    % hardware FIFO when the FIFO is full
    %set(slaveDeviceObj.Analoginputacquisition(1), 'WrapEnabled', 'off');
    
    % Set the slave clock source to internal
    set(slaveDeviceObj.Referenceoscillator(1), 'ReferenceClockSource', 'DT8824ReferenceOscillatorSourceInternal');
    
    % Set the master clock source to internal
    set(masterDeviceObj.Referenceoscillator(1), 'ReferenceClockSource', 'DT8824ReferenceOscillatorSourceInternal');
    
    % Enable the master instrument to drive the reference clock on the trigger bus
    set(masterDeviceObj.Referenceoscillator(1), 'OutputEnabled', 'on');
    
    % Set the slave clock source to LXI line 7 on the trigger bus
    set(slaveDeviceObj.Referenceoscillator(1), 'ReferenceClockSource', 'DT8824ReferenceOscillatorSourceLXI7');
    
    % set slave the clock frequency
    set(slaveDeviceObj.Analoginputacquisition(1), 'SampleRate', 1000);
    
    % Set master the clock frequency
    set(masterDeviceObj.Analoginputacquisition(1), 'SampleRate', 1000);
    
    
    % Set the trigger source to start streaming when the instrument
    % recievs the Initiate command
    set(masterDeviceObj.Analoginputtrigger(1), 'Source', 'IMMEdiate');
    
    % Set the master trigger bus line1 (LXI0) state to be driven.  This MUST be done
    % prior to setting the slave trigger source to avoid false triggers.
    set(masterDeviceObj.Triggerbusline(1), 'Enabled', 'on');
    
    % Set the slave trigger bus line1 (LXI0) state to be disabled.
    set(slaveDeviceObj.Triggerbusline(1), 'Enabled', 'off');
    
    % Set the slave trigger source to the value that corresponds to
    % masterDeviceObj.Triggerbusline(1) where
    % masterDeviceObj.Triggerbusline(1) = 'LXI0'
    % masterDeviceObj.Triggerbusline(2) = 'LXI1'
    % masterDeviceObj.Triggerbusline(3) = 'LXI2'
    % masterDeviceObj.Triggerbusline(4) = 'LXI3'
    % masterDeviceObj.Triggerbusline(5) = 'LXI4'
    % masterDeviceObj.Triggerbusline(6) = 'LXI5'
    set(slaveDeviceObj.Analoginputtrigger(1), 'Source', 'LXI0');
    
    %% Initiate the acquisition on the slave instrument.
    
    % Now, the slave leaves the idle state and waits for the trigger on
    % LXI0 trigger bus line
    groupObj = get(slaveDeviceObj, 'Analoginputacquisition');
    slaveAnalogInputAcquisitionObj = groupObj(1);
    invoke(slaveAnalogInputAcquisitionObj, 'Arm');
    invoke(slaveAnalogInputAcquisitionObj, 'Initiate');
    
    %% Initiate the acquisition on the master instrument.
    
    
    
    % Now, the slave leaves the 'waiting for trigger' state and starts
    % acquiring data
    masterGroupObj = get(masterDeviceObj, 'Analoginputacquisition');
    masterAnalogInputAcquisitionObj = masterGroupObj(1);
    invoke(masterAnalogInputAcquisitionObj, 'Arm');
    invoke(masterAnalogInputAcquisitionObj, 'Initiate');
        
    isRunning = false;
    ScanIndex = 0;
    RequestedScansToRead = 100;
    
    %% Process data for the master instrument
    
    while (isRunning == false || ScanIndex < (RequestedScansToRead-1))
    [ScanIndex, isRunning, isArmed, isTriggered, isADSyncDetected, isADFifoOverflow] = invoke(masterAnalogInputAcquisitionObj, 'GetStatus', 0, 0, 0, 0, 0, 0);
    end
    
    % Read 100 scans from the master instrument
    RequestedScansIndex = 0;
    [ActualScansIndex, ActualScansRead, StartTimeInSeconds, StartTimeInMilliSeconds, masterSamples] = invoke(masterAnalogInputAcquisitionObj, 'Fetch', int32(RequestedScansIndex), int32(RequestedScansToRead), int32(0), int32(0), int32(0), int32(0), [0;0]);
    
    RequestedScansIndex = ActualScansIndex+ActualScansRead;
    disp(['Master: Actual Scans Index: ', num2str(ActualScansIndex) ,'  Actual Scans Read: ', num2str(ActualScansRead)]);
    disp(masterSamples);
    
    %% Process data for the slave instrument
    
    isRunning = false;
    ScanIndex = 0;
    
    while (isRunning == false || ScanIndex < (RequestedScansToRead-1))
    [ScanIndex, isRunning, isArmed, isTriggered, isADSyncDetected, isADFifoOverflow] = invoke(slaveAnalogInputAcquisitionObj, 'GetStatus', 0, 0, 0, 0, 0, 0);
    end
    
    % Read 100 scans from the slave
    RequestedScansIndex = 0;
    [ActualScansIndex, ActualScansRead, StartTimeInSeconds, StartTimeInMilliSeconds, slaveSamples] = invoke(slaveAnalogInputAcquisitionObj, 'Fetch', int32(RequestedScansIndex), int32(RequestedScansToRead), int32(0), int32(0), int32(0), int32(0), [0;0]);
    
    RequestedScansIndex = ActualScansIndex+ActualScansRead;
    disp(['slave: Actual Scans Index: ', num2str(ActualScansIndex) ,'  Actual Scans Read: ', num2str(ActualScansRead)]);
    
    %% Plot the data
    
    disp(slaveSamples);
    hold on;
    plot(masterSamples);
    
    plot(slaveSamples,'green');
    hold off;
    
    %% Stop the acquisition
    
    invoke(masterAnalogInputAcquisitionObj, 'Abort');
    invoke(slaveAnalogInputAcquisitionObj, 'Abort');
    
 
catch DT8824error
    disp(['Error id: ', DT8824error.identifier]);
    disp(['Error Message: ',DT8824error.message]);
end

%% Disconnect the device objects from the instruments and remove them from memory

disconnect(masterDeviceObj);
delete(masterDeviceObj);   

disconnect(slaveDeviceObj);
delete(slaveDeviceObj);   
